<?php

namespace App\Services\auth;
use App\Models\User;
use App\Services\sms\SmsFarazService;
use Carbon\Carbon;
class OtpService
{
    protected $smsService;

    public function __construct(SmsFarazService $smsService)
    {
        $this->smsService = $smsService;
    }

    /**
     * تولید و ذخیره کد OTP برای کاربر
     *
     * @param User $user
     * @return int کد OTP تولید شده
     */
    public function generateAndSaveOtp(User $user): int
    {
        $otp = rand(100000, 999999);
        $user->otp_code = $otp;
        $user->otp_code_expires_at = now()->addMinutes(5);
        $user->otp_attempts = 0;
        $user->save();

        return $otp;
    }

    /**
     * ارسال پیامک OTP به شماره موبایل کاربر
     *
     * @param User $user
     * @return \Illuminate\Http\Client\Response
     */
    public function sendOtpSms(User $user)
    {
        $patternCode = 'kric7d5dkxju5tw'; // کد الگو پیامک (در صورت نیاز تغییر دهید)
        $sender = '+983000505';            // شماره فرستنده
        $recipient = $user->mobile;        // شماره گیرنده
        $variables = ['verification-code' => (string)$user->otp_code];

        return $this->smsService->sendPatternSms($patternCode, $sender, $recipient, $variables);
    }

    /**
     * تولید، ذخیره و ارسال OTP
     *
     * @param User $user
     * @return \Illuminate\Http\Client\Response
     */
    public function generateAndSendOtp(User $user)
    {
        $this->generateAndSaveOtp($user);
        return $this->sendOtpSms($user);
    }

    public function verifyOtp(string $mobile, string $otpCode): array
    {
        $user = User::where('mobile', $mobile)->first();

        if (!$user) {
            return ['status' => 'error', 'message' => 'کاربر یافت نشد', 'code' => 404];
        }

        if ($user->otp_attempts >= 5) {
            return ['status' => 'error', 'message' => 'تعداد تلاش‌های شما برای وارد کردن کد به پایان رسیده است. لطفا دوباره کد بگیرید.', 'code' => 429];
        }

        if (!$user->otp_code_expires_at || Carbon::parse($user->otp_code_expires_at)->isPast()) {
            return ['status' => 'error', 'message' => 'کد منقضی شده است. لطفا دوباره کد بگیرید.', 'code' => 410];
        }

        if ($user->otp_code !== $otpCode) {
            $user->increment('otp_attempts');
            return ['status' => 'error', 'message' => 'کد اشتباه است', 'code' => 422];
        }

        // تایید موفق
        $user->mobile_verified_at = now();
        $user->otp_code = null;
        $user->otp_code_expires_at = null;
        $user->otp_attempts = 0;
        $user->save();

        // ایجاد توکن
        $token = $user->createToken('auth_token')->plainTextToken;

        return [
            'status' => 'success',
            'message' => 'ورود موفقیت‌آمیز بود',
            'access_token' => $token,
            'token_type' => 'Bearer',
        ];
    }
}
