<?php

namespace App\Http\Controllers\dashboard\ticket;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Services\ticket\TicketService;
use App\Http\Requests\TicketStoreRequest;
use App\Http\Requests\TicketReplyStoreRequest;
use App\Services\ticket\TicketReplyService;
use Auth;
class TicketController extends Controller
{
    protected $ticketService;
    protected $ticketReplyService;
    public function __construct(TicketService $ticketService,TicketReplyService $ticketReplyService)
    {
        $this->ticketService = $ticketService;
        $this->ticketReplyService = $ticketReplyService;
    }

       public function store(TicketStoreRequest $request): JsonResponse
{
    $timestamp = now()->toIso8601String();
    $version = 'v1';

    $user = Auth::user();

    $ticket = $this->ticketService->createTicket(
        $user->id,
        $request->only(['department_id', 'priority_id', 'subject', 'description', 'assigned_to']),
        $request->file('attachments')
    );

    return response()->json([
        'status' => 'success',
        'message' => 'تیکت با موفقیت ایجاد شد.',
        'data' => $ticket->load(['department', 'priority', 'attachments']),
        'meta' => [
            'timestamp' => $timestamp,
            'version' => $version
        ]
    ]);
}

public function getUserTickets(Request $request): JsonResponse
{
    $timestamp = now()->toIso8601String();
    $version = 'v1';

    $user = Auth::user();

    if (!$user) {
        return response()->json([
            'status' => 'error',
            'message' => 'Unauthorized',
            'meta' => [
                'timestamp' => $timestamp,
                'version' => $version
            ]
        ], 401);
    }

    $priority = $request->query('priority', null);

    $tickets = $this->ticketService->getUserTickets($user->id, $priority)->map(function ($ticket) {
        return [
            'id' => $ticket->id,
            'subject' => $ticket->subject,
            'department' => $ticket->department?->name,
            'priority' => $ticket->priority?->name,
            'status' => $ticket->status,
            'created_at' => $ticket->created_at->toDateTimeString(),
        ];
    });

    return response()->json([
        'status' => 'success',
        'message' => 'Tickets retrieved successfully',
        'data' => $tickets,
        'meta' => [
            'timestamp' => $timestamp,
            'version' => $version
        ]
    ]);
}

public function showTicket($id, Request $request)
{
    $timestamp = now()->toIso8601String();
    $version = 'v1';

    $user = Auth::user();

    if (!$user) {
        return response()->json([
            'status' => 'error',
            'message' => 'Unauthorized',
            'meta' => [
                'timestamp' => $timestamp,
                'version' => $version
            ]
        ], 401);
    }

    // گرفتن تیکت فقط اگر متعلق به کاربر باشه
    $ticket = $this->ticketService->getTicketByIdAndUser($id, $user->id);

    if (!$ticket) {
        return response()->json([
            'status' => 'error',
            'message' => 'Ticket not found or access denied',
            'meta' => [
                'timestamp' => $timestamp,
                'version' => $version
            ]
        ], 404);
    }

    return response()->json([
        'status' => 'success',
        'message' => 'Ticket retrieved successfully',
        'data' => [
            'ticket' => $ticket,
            'replies' => $ticket->replies // فرض بر اینکه رابطه replies تعریف شده
        ],
        'meta' => [
            'timestamp' => $timestamp,
            'version' => $version
        ]
    ]);
}

public function replies(TicketReplyStoreRequest $request): JsonResponse
{
    $timestamp = now()->toIso8601String();
    $version = 'v1';

    $user = auth()->user();

    $reply = $this->ticketReplyService->createReply(
        $user->id,
        $request->only(['ticket_id', 'message']),
        $request->file('attachments')
    );

    return response()->json([
        'status' => 'success',
        'message' => 'پاسخ تیکت با موفقیت ثبت شد.',
        'data' => $reply->load(['attachments']),
        'meta' => [
            'timestamp' => $timestamp,
            'version' => $version
        ]
    ]);
}

public function updateTicketStatus($id)
{
    $timestamp = now()->toIso8601String();
    $version = 'v1';

    $user = Auth::user();

    if (!$user) {
        return response()->json([
            'status' => 'error',
            'message' => 'Unauthorized',
            'meta' => [
                'timestamp' => $timestamp,
                'version' => $version
            ]
        ], 401);
    }

    $ticket = Ticket::where('id', $id)
        ->where('user_id', $user->id)
        ->first();

    if (!$ticket) {
        return response()->json([
            'status' => 'error',
            'message' => 'Ticket not found or access denied',
            'meta' => [
                'timestamp' => $timestamp,
                'version' => $version
            ]
        ], 404);
    }

    $ticket->status = 'closed';  // مقدار مورد نظر
    $ticket->save();

    return response()->json([
        'status' => 'success',
        'message' => 'Ticket status updated successfully',
        'meta' => [
            'timestamp' => $timestamp,
            'version' => $version
        ]
    ]);
}


}
